// SJJ Embedded Micro Solutions, LLC.
// Copyright  2009 SJJ Embedded Micro Solutions, LLC. All Rights Reserved
//  
// THIS SAMPLE CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, 
// INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.

/*****************************************************************************
*
* FILE NAME:    RegionPriInvert.c
*
* DESCRIPTION:    This is the main program module.
*
\*****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <rt.h>


#include "RegionPriInvert.h"

// global variables
  RTHANDLE hRootProcess;
  DWORD    dwKtickInUsecs;

//++
  // Mutual exclusion region handle
  RTHANDLE  hMutex;

  RTHANDLE  hClientThread1event;
  RTHANDLE  hClientThread2event;
  RTHANDLE  hClientThread3event;
//++

// client thread declarations

  void     ClientThread1(void);

  void     ClientThread2(void);

  void     ClientThread3(void);

// module variables
struct _INIT {
  enum States
       { BEFORE_INIT,
         INIT_BUSY,
         INIT_DONE,
         CLEANUP_BUSY } state;
  RTHANDLE hMain;
  BOOL     bCataloged;
	// client thread declarations

  RTHANDLE hClientThread1;

  RTHANDLE hClientThread2;

  RTHANDLE hClientThread3;

}  strInit;

static void     Cleanup(void);

/*****************************************************************************
*
* FUNCTION:     main
*
* DESCRIPTION:
*  This is the main program module.
*  It creates shared memory objects, and all threads.
*  The main thread then waits for a deletion message,
*  and then all objects are deleted and the process terminates itself.
*
\*****************************************************************************/

void        main(void)
{
  SYSINFO   sysinfo;
  EVENTINFO eiEventInfo;

//++
  BYTE bThreadPriority;
//++

#ifdef _DEBUG
  printf("RegionPriInvert started\n");
#endif

  // obtain handle of root process (cannot fail)
  hRootProcess = GetRtThreadHandles(ROOT_PROCESS);

  // initialize the structure for cleaning up
  memset(&strInit, 0, sizeof(strInit));
  strInit.state = BEFORE_INIT;
  
  // get low level tick length in usecs
  if (!CopyRtSystemInfo(&sysinfo))
    Fail("Cannot copy system info");
  dwKtickInUsecs = 10000 / sysinfo.KernelTickRatio;
  if (dwKtickInUsecs == 0)
    Fail("Invalid low level tick length");

  // adjust process max priority (ignore error)
  SetRtProcessMaxPriority(NULL_RTHANDLE, 150);

  // obtain main thread's handle
  strInit.hMain = GetRtThreadHandles(THIS_THREAD);
  strInit.state = INIT_BUSY;

//++
#ifdef _DEBUG
  printf("Main Process Thread Handle: 0x%x\n", strInit.hMain);
#endif
//++

  // catalog the token of this process in the root process
  if (!Catalog(hRootProcess,
      GetRtThreadHandles(THIS_PROCESS),
      "RegPriInvert"))
    Fail("Cannot catalog process name");
  strInit.bCataloged = TRUE;

  //++
  // Create the mutual exclusion region
  // Use priority-based thread queuing for priority inversion protection
  hMutex = CreateRtRegion(PRIORITY_QUEUING);
  if (hMutex == BAD_RTHANDLE)
  {
    Fail("Cannot create region hMutex");
  }

  // Get access to region
  if (!AcceptRtControl(hMutex))
  {
    Fail("Cannot get access to region");
  }

  // Get current thread's priority
  bThreadPriority = GetRtThreadPriority(NULL_RTHANDLE);

  // Create event semaphore for ClientThread1
  hClientThread1event = CreateRtSemaphore(0, 1, FIFO_QUEUING);
  if (hClientThread1event == BAD_RTHANDLE)
  {
    Fail("Cannot create semaphore hClientThread1event");
  }

  // Create event semaphore for ClientThread2
  hClientThread2event = CreateRtSemaphore(0, 1, FIFO_QUEUING);
  if (hClientThread2event == BAD_RTHANDLE)
  {
    Fail("Cannot create semaphore hClientThread2event");
  }

  // Create event semaphore for ClientThread3
  hClientThread3event = CreateRtSemaphore(0, 1, FIFO_QUEUING);
  if (hClientThread3event == BAD_RTHANDLE)
  {
    Fail("Cannot create semaphore hClientThread3event");
  }

//++

  // create client threads
//+-
  //hClientThread1 has higher priority than the main process thread but lower than hClientThread2 or hClientThread3

  strInit.hClientThread1
          = CreateRtThread((bThreadPriority - 1),
            ClientThread1,
            4096,
            0);
//+-

  if (strInit.hClientThread1 == BAD_RTHANDLE)
  {
    strInit.hClientThread1 = NULL_RTHANDLE;
    Fail("Cannot create client thread ClientThread1");
  }

//++
#ifdef _DEBUG
  printf("hClientThread1 Handle: 0x%x\n", strInit.hClientThread1);
#endif
//++

//+-
  //hClientThread2 is higher priority than the main process thread, higher than hClientThread1, but lower than hClientThread3
  strInit.hClientThread2
          = CreateRtThread((bThreadPriority - 2),
            ClientThread2,
            4096,
            0);
//+-

  if (strInit.hClientThread2 == BAD_RTHANDLE)
  {
    strInit.hClientThread2 = NULL_RTHANDLE;
    Fail("Cannot create client thread ClientThread2");
  }

//++
#ifdef _DEBUG
  printf("hClientThread2 Handle: 0x%x\n", strInit.hClientThread2);
#endif
//++

//+-
  //ThreadC has the highest priority
  strInit.hClientThread3
          = CreateRtThread((bThreadPriority - 3),
            ClientThread3,
            4096,
            0);
//+-

  if (strInit.hClientThread3 == BAD_RTHANDLE)
  {
    strInit.hClientThread3 = NULL_RTHANDLE;
    Fail("Cannot create client thread ClientThread3");
  }

//++
#ifdef _DEBUG
  printf("hClientThread3 Handle: 0x%x\n", strInit.hClientThread3);
#endif
//++

  // indicate that initialization has finished
  strInit.state = INIT_DONE;
#ifdef _DEBUG
  printf("RegionPriInvert finished initialization\n");
#endif

//++
  //Start ClientThread1
  printf("Starting priority inversion sequence\n");

  // First release the mutex region
  if ( !ReleaseRtControl() )
  {
    Fail("Main process thread failed to release region!");
  }

  // Now release unit to ThreadA event semaphore
  if ( !ReleaseRtSemaphore(hClientThread1event, 1) )
  {
    Fail("Main process thread failed to release hClientThread1event semaphore!");
  }
//++
  // wait for notification
  while (RtNotifyEvent(RT_SYSTEM_NOTIFICATIONS | RT_EXIT_NOTIFICATIONS,
      WAIT_FOREVER,
      &eiEventInfo))
  {
    switch(eiEventInfo.dwNotifyType)
    {
    case TERMINATE:
      // TODO: this process should terminate
      // cleanup the environment
      Cleanup();  // does not return

    case RT_CLIENT_DOWN:
      // TODO: on a NT host react to a RT client that has gone down
      break;

    case RT_CLIENT_UP:
      // TODO: on a NT host react to a RT client that has come back
      break;

    case NT_HOST_DOWN:
      // TODO: on a RT client react to a NT host that has gone down
      break;

    case NT_HOST_UP:
      // TODO: on a RT client react to a NT host that has come back
      break;

    case NT_BLUESCREEN:
      // TODO: react to a NT blue screen
      break;
    }
  }
  Fail("Notify failed");
}


/*****************************************************************************
*
* FUNCTION:     Fail
*
* PARAMETERS:   same parameters as expected by printf
*
* RETURNS:      does not return
*
* DESCRIPTION:
*  If in debug mode, prints the message.
*  Then the current process is killed ungraciously.
\*****************************************************************************/

void        Fail(
  LPSTR       lpszMessage, ...)
{
  EXCEPTION   eh;
  RTHANDLE    hDelMbx;
  DWORD     dwTerminate;

#ifdef _DEBUG
  va_list     ap;

  va_start(ap, lpszMessage);
  vprintf(lpszMessage, ap);
  va_end(ap);
  printf("\nError nr=%x\n", GetLastRtError());
#endif

  // make sure that exceptions are returned for inline handling
  GetRtExceptionHandlerInfo(THREAD_HANDLER, &eh);
  eh.ExceptionMode = 0;
  SetRtExceptionHandler(&eh);

  // if we had not started initializing yet, just get out
  if (strInit.state == BEFORE_INIT)
    exit(0);

  if (strInit.hMain == GetRtThreadHandles(THIS_THREAD))
  {
    // this is the main thread:
    // if we are busy initializing, then do Cleanup
    if (strInit.state == INIT_BUSY)
      Cleanup();  // does not return

    // this is the main thread, but we are not initializing:
    // just return
    return;
  }

  // this is not the main thread:
  // ask main thread to do cleanup
  // (allow some time to setup the deletion mailbox, ignore errors)
  hDelMbx = LookupRtHandle(NULL_RTHANDLE, "R?EXIT_MBOX", 5000);
  dwTerminate = TERMINATE;
  SendRtData(hDelMbx, &dwTerminate, 4);
  SuspendRtThread(NULL_RTHANDLE);

  // if we get here, suspend has failed
  while (1)
    RtSleep(655349);  // sleep for the maximum time
}


/*****************************************************************************
*
* FUNCTION:   Cleanup (local function)
*
* DESCRIPTION:
*  Remove all objects and other process side effects, 
*  as far as they have been created.
*
\*****************************************************************************/

void      Cleanup(void)
{
  // indicate that we are cleaning up
  strInit.state = CLEANUP_BUSY;

  // kill client threads

  if (strInit.hClientThread1 != NULL_RTHANDLE)
    if (!DeleteRtThread(strInit.hClientThread1))
      Fail("Cannot delete client thread ClientThread1");

  if (strInit.hClientThread2 != NULL_RTHANDLE)
    if (!DeleteRtThread(strInit.hClientThread2))
      Fail("Cannot delete client thread ClientThread2");

  if (strInit.hClientThread3 != NULL_RTHANDLE)
    if (!DeleteRtThread(strInit.hClientThread3))
      Fail("Cannot delete client thread ClientThread3");
//++
  // Remove region and semaphores
  if (hMutex != NULL_RTHANDLE)
  {
    if (!DeleteRtRegion(hMutex))
    {
      Fail("Cannot delete mutex region");
    }
  }

  if (hClientThread1event != NULL_RTHANDLE)
  {
    if (!DeleteRtSemaphore(hClientThread1event))
    {
      Fail("Cannot delete event semaphore hClientThread1event");
    }
  }

  if (hClientThread2event != NULL_RTHANDLE)
  {
    if (!DeleteRtSemaphore(hClientThread2event))
    {
      Fail("Cannot delete event semaphore hClientThread2event");
    }
  }

  if (hClientThread3event != NULL_RTHANDLE)
  {
    if (!DeleteRtSemaphore(hClientThread3event))
    {
      Fail("Cannot delete event semaphore hClientThread3event");
    }
  }



//++
  // remove our name from the root process
  if (strInit.bCataloged)
    if (!UncatalogRtHandle(hRootProcess, "RegionPriInvert"))
      Fail("Cannot remove my own name");

#ifdef _DEBUG
  printf("RegionPriInvert finished cleaning up\n");
#endif

  // lie down
  exit(0);
}
