// SJJ Embedded Micro Solutions, LLC.
// Copyright  2009 SJJ Embedded Micro Solutions, LLC. All Rights Reserved
//  
// THIS SAMPLE CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, 
// INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.

/*****************************************************************************
*
* FILE NAME:    ParallelRT.c
*
* DESCRIPTION:    This is the main program module.
*
\*****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <rt.h>


#include "ParallelRT.h"

// global variables
  RTHANDLE hRootProcess;
  DWORD    dwKtickInUsecs;
// interrupt handler declarations

  BOOL     InterruptHndlr1Init(void);
  BOOL     InterruptHndlr1Kill(void);

  //parallel port registers
  int portCSR;
  int portCSRnew;
  int portECP_ECR;
  int portECP_ECRnew;

// module variables
struct _INIT {
  enum States
       { BEFORE_INIT,
         INIT_BUSY,
         INIT_DONE,
         CLEANUP_BUSY } state;
  RTHANDLE hMain;
  BOOL     bCataloged;
	// interrupt handler declarations

  BOOL     bInterruptHndlr1;

}  strInit;

static void     Cleanup(void);

/*****************************************************************************
*
* FUNCTION:     main
*
* DESCRIPTION:
*  This is the main program module.
*  It creates shared memory objects, and all threads.
*  The main thread then waits for a deletion message,
*  and then all objects are deleted and the process terminates itself.
*
\*****************************************************************************/

void        main(void)
{
  SYSINFO   sysinfo;
  EVENTINFO eiEventInfo;
#ifdef _DEBUG
  printf("ParallelRT started\n");
#endif

  // obtain handle of root process (cannot fail)
  hRootProcess = GetRtThreadHandles(ROOT_PROCESS);

  // initialize the structure for cleaning up
  memset(&strInit, 0, sizeof(strInit));
  strInit.state = BEFORE_INIT;
  
  // get low level tick length in usecs
  if (!CopyRtSystemInfo(&sysinfo))
    Fail("Cannot copy system info");
  dwKtickInUsecs = 10000 / sysinfo.KernelTickRatio;
  if (dwKtickInUsecs == 0)
    Fail("Invalid low level tick length");

  // adjust process max priority (ignore error)
  SetRtProcessMaxPriority(NULL_RTHANDLE, 150);

  // obtain main thread's handle
  strInit.hMain = GetRtThreadHandles(THIS_THREAD);
  strInit.state = INIT_BUSY;

  // catalog the token of this process in the root process
  if (!Catalog(hRootProcess,
      GetRtThreadHandles(THIS_PROCESS),
      "ParallelRT"))
    Fail("Cannot catalog process name");
  strInit.bCataloged = TRUE;

 //First set ECP port to bidirectional
 //Setup Parallel port for data line input and interrupt on ACK

  portECP_ECR = _inp(0x77A);
  portECP_ECRnew = portECP_ECR | 0x20;
  _outp(0x77A, portECP_ECRnew);


  portCSR = _inp(0x37A);
  portCSRnew = portCSR | 0x30;
  _outp(0x37A, portCSRnew);
 

  // create interrupt handlers

  if (!InterruptHndlr1Init())
    Fail("Cannot initialize interrupt IRQ7");
  strInit.bInterruptHndlr1 = TRUE;

  // indicate that initialization has finished
  strInit.state = INIT_DONE;
#ifdef _DEBUG
  printf("ParallelRT finished initialization\n");
#endif

  // wait for notification
  while (RtNotifyEvent(RT_SYSTEM_NOTIFICATIONS | RT_EXIT_NOTIFICATIONS,
      WAIT_FOREVER,
      &eiEventInfo))
  {
    switch(eiEventInfo.dwNotifyType)
    {
    case TERMINATE:
      // TODO: this process should terminate
      // cleanup the environment
      Cleanup();  // does not return

    case RT_CLIENT_DOWN:
      // TODO: on a NT host react to a RT client that has gone down
      break;

    case RT_CLIENT_UP:
      // TODO: on a NT host react to a RT client that has come back
      break;

    case NT_HOST_DOWN:
      // TODO: on a RT client react to a NT host that has gone down
      break;

    case NT_HOST_UP:
      // TODO: on a RT client react to a NT host that has come back
      break;

    case NT_BLUESCREEN:
      // TODO: react to a NT blue screen
      break;
    }
  }
  Fail("Notify failed");
}


/*****************************************************************************
*
* FUNCTION:     Fail
*
* PARAMETERS:   same parameters as expected by printf
*
* RETURNS:      does not return
*
* DESCRIPTION:
*  If in debug mode, prints the message.
*  Then the current process is killed ungraciously.
\*****************************************************************************/

void        Fail(
  LPSTR       lpszMessage, ...)
{
  EXCEPTION   eh;
  RTHANDLE    hDelMbx;
  DWORD     dwTerminate;

#ifdef _DEBUG
  va_list     ap;

  va_start(ap, lpszMessage);
  vprintf(lpszMessage, ap);
  va_end(ap);
  printf("\nError nr=%x\n", GetLastRtError());
#endif

  // make sure that exceptions are returned for inline handling
  GetRtExceptionHandlerInfo(THREAD_HANDLER, &eh);
  eh.ExceptionMode = 0;
  SetRtExceptionHandler(&eh);

  // if we had not started initializing yet, just get out
  if (strInit.state == BEFORE_INIT)
    exit(0);

  if (strInit.hMain == GetRtThreadHandles(THIS_THREAD))
  {
    // this is the main thread:
    // if we are busy initializing, then do Cleanup
    if (strInit.state == INIT_BUSY)
      Cleanup();  // does not return

    // this is the main thread, but we are not initializing:
    // just return
    return;
  }

  // this is not the main thread:
  // ask main thread to do cleanup
  // (allow some time to setup the deletion mailbox, ignore errors)
  hDelMbx = LookupRtHandle(NULL_RTHANDLE, "R?EXIT_MBOX", 5000);
  dwTerminate = TERMINATE;
  SendRtData(hDelMbx, &dwTerminate, 4);
  SuspendRtThread(NULL_RTHANDLE);

  // if we get here, suspend has failed
  while (1)
    RtSleep(655349);  // sleep for the maximum time
}


/*****************************************************************************
*
* FUNCTION:   Cleanup (local function)
*
* DESCRIPTION:
*  Remove all objects and other process side effects, 
*  as far as they have been created.
*
\*****************************************************************************/

void      Cleanup(void)
{
  // indicate that we are cleaning up
  strInit.state = CLEANUP_BUSY;

  // terminate interrupt handlers

  if (strInit.bInterruptHndlr1)
    if (!InterruptHndlr1Kill())
      Fail("Cannot terminate interrupt IRQ7");

  //Return parallel ECP and CSR to the orignal values
  _outp(0x37A, portCSR);
  _outp(0x77A, portECP_ECR);


  // remove our name from the root process
  if (strInit.bCataloged)
    if (!UncatalogRtHandle(hRootProcess, "ParallelRT"))
      Fail("Cannot remove my own name");

#ifdef _DEBUG
  printf("ParallelRT finished cleaning up\n");
#endif

  // lie down
  exit(0);
}
